// OpenCV Core (4.5.1 兼容版本 - 简化版)
var Mat = org.opencv.core.Mat;
var Rect = org.opencv.core.Rect;
var Size = org.opencv.core.Size;
var CvType = org.opencv.core.CvType;
var Scalar = org.opencv.core.Scalar;
var Point = org.opencv.core.Point;
var Bitmap = android.graphics.Bitmap

// OpenCV 其他模块
var Utils = org.opencv.android.Utils;
var Imgproc = org.opencv.imgproc.Imgproc;
var Core = org.opencv.core.Core;

/**
 * 简化版滤镜映射表（移除反色和锐化）
 */
var filterMap = {
    // 基础滤镜
    "灰度滤镜": "grayscale",
    "黑白滤镜": "binary",
    "自适应二值化": "adaptive_binary",
    "轮廓滤镜": "edge",

    // 模糊滤镜
    "高斯模糊": "gaussian_blur",
    "均值模糊": "box_blur",
    "中值模糊": "median_blur",

    // 色彩调整
    "高对比度": "high_contrast",
    "亮度调整": "brightness",
    "颜色替换": "replacecolor",

    // 特殊效果
    "素描效果": "sketch",

    // 形态学操作
    "腐蚀": "erosion",
    "膨胀": "dilation",
    "开运算": "opening",
    "闭运算": "closing",
};

/**
 * 简化版默认参数配置
 */
var defaultConfigs = {
    // 基础滤镜
    grayscale: {},
    binary: {
        threshold: 127,
        maxValue: 255,
        type: Imgproc.THRESH_BINARY
    },
    adaptive_binary: {
        maxValue: 255,
        adaptiveMethod: Imgproc.ADAPTIVE_THRESH_MEAN_C,
        thresholdType: Imgproc.THRESH_BINARY,
        blockSize: 11,
        C: 2
    },
    edge: {
        threshold1: 50,
        threshold2: 150,
        apertureSize: 3,
        L2gradient: false
    },

    // 模糊滤镜
    gaussian_blur: {
        kernelSize: 15,
        sigmaX: 0,
        sigmaY: 0
    },
    box_blur: {
        kernelSize: 15
    },
    median_blur: {
        kernelSize: 15
    },

    // 色彩调整
    high_contrast: {
        alpha: 1.5,
        beta: 0
    },
    brightness: {
        value: 30
    },

    // 特殊效果
    sketch: {},
    
    // 颜色替换
    replacecolor: { 
        colors: [
        ], 
        other: "" 
    },

    // 形态学操作
    erosion: {
        kernelSize: 5,
        iterations: 1,
        shape: Imgproc.MORPH_RECT
    },
    dilation: {
        kernelSize: 5,
        iterations: 1,
        shape: Imgproc.MORPH_RECT
    },
    opening: {
        kernelSize: 5,
        iterations: 1,
        shape: Imgproc.MORPH_RECT
    },
    closing: {
        kernelSize: 5,
        iterations: 1,
        shape: Imgproc.MORPH_RECT
    },
};

/**
 * 获取形态学核
 */
function getMorphologyKernel(size, shape) {
    try {
        return Imgproc.getStructuringElement(shape, new Size(size, size));
    } catch (e) {
        console.log("创建形态学核失败: " + e);
        return Imgproc.getStructuringElement(Imgproc.MORPH_RECT, new Size(5, 5));
    }
}

/**
 * 安全的颜色空间转换
 */
function safeColorConvert(src, dst, code) {
    try {
        if (src.channels() == 4 && (code == Imgproc.COLOR_BGR2GRAY || code == Imgproc.COLOR_RGB2GRAY)) {
            code = Imgproc.COLOR_BGRA2GRAY;
        }
        if (src.channels() == 1 && (code == Imgproc.COLOR_GRAY2BGR || code == Imgproc.COLOR_GRAY2RGB)) {
            code = Imgproc.COLOR_GRAY2BGRA;
        }
        Imgproc.cvtColor(src, dst, code);
        return true;
    } catch (e) {
        console.log("颜色空间转换失败: " + e);
        src.copyTo(dst);
        return false;
    }
}

/**
 * 检查Mat是否为空或无效的辅助函数
 */
function isMatEmpty(mat) {
    if (!mat) return true;

    try {
        // 检查Mat的基本属性
        var cols = mat.cols();
        var rows = mat.rows();

        // 检查尺寸是否有效
        if (cols <= 0 || rows <= 0) {
            return true;
        }

        // 使用OpenCV的empty()方法作为最终检查
        return mat.empty();

    } catch (e) {
        console.log("检查Mat状态时异常: " + e.message);
        return true;
    }
}

/**
 * 安全的log函数
 */
function log(message) {
    try {
        if (typeof console !== 'undefined' && console.log) {
            console.log(message);
        }
    } catch (e) {
        // 忽略日志异常
    }
}

/**
 * 获取默认输出
 */
function getDefaultOutput(type) {
    switch (type) {
        case "mat":
            return new Mat();
        case "bitmap":
            return null;
        default:
            return images.createImage(1, 1);
    }
}

/**
 * Mat输出处理
 */
function handleMatOutput(input) {
    if (input instanceof Mat) {
        // 检查Mat是否有效（检查尺寸和数据地址）
        if (isMatEmpty(input)) {
            console.log("输入Mat为空或无效 - 尺寸: " + input.cols() + "x" + input.rows());
            return new Mat();
        }
        return input.clone();
    }

    var tmp = new Mat();
    try {
        if (input instanceof Bitmap) {
            // 检查Bitmap是否已回收
            if (input.isRecycled()) {
                console.log("Bitmap已被回收，无法转换");
                return new Mat();
            }
            Utils.bitmapToMat(input, tmp);
        } else if (input && typeof input.getBitmap === 'function') {
            var bitmap = input.getBitmap();
            if (bitmap && !bitmap.isRecycled()) {
                Utils.bitmapToMat(bitmap, tmp);
            } else {
                console.log("获取的Bitmap无效或已回收");
                tmp.release();
                return new Mat();
            }
        } else {
            console.log("无法转换到Mat: 不支持的输入类型");
            tmp.release();
            return new Mat();
        }
        return tmp;
    } catch (e) {
        console.log("Mat转换异常: " + e.message);
        if (tmp && !isMatEmpty(tmp)) {
            tmp.release();
        }
        return new Mat();
    }
}

/**
 * Bitmap输出处理
 */
function handleBitmapOutput(input) {
    if (input instanceof Bitmap) {
        if (input.isRecycled()) {
            console.log("输入Bitmap已被回收");
            return null;
        }
        return input;
    }

    if (input && typeof input.getBitmap === 'function') {
        var bitmap = input.getBitmap();
        if (bitmap && !bitmap.isRecycled()) {
            return bitmap;
        } else {
            console.log("获取的Bitmap无效或已回收");
            return null;
        }
    }

    if (input instanceof Mat) {
        return matToBitmap(input);
    }

    console.log("无法转换到Bitmap: 不支持的输入类型");
    return null;
}

/**
 * Mat转Bitmap的安全转换
 */
function matToBitmap(mat) {
    log("Mat信息: " + mat);
    if (!mat || isMatEmpty(mat)) {
        console.log("Mat为空或无效，无法转换为Bitmap - 尺寸: " +
            (mat ? (mat.cols() + "x" + mat.rows()) : "null"));
        return null;
    }

    var outputMat = new Mat();
    var bmp = null;

    try {
        // 根据通道数进行颜色空间转换
        var channels = mat.channels();
        log("Mat信息 - 类型: " + mat.type() + ", 通道数: " + channels +
            ", 尺寸: " + mat.cols() + "x" + mat.rows());

        if (channels === 1) {
            log("单通道转换");
            Imgproc.cvtColor(mat, outputMat, Imgproc.COLOR_GRAY2BGRA);
        } else if (channels === 3) {
            log("3通道转换");
            Imgproc.cvtColor(mat, outputMat, Imgproc.COLOR_BGR2BGRA);
        } else if (channels === 4) {
            log("4通道直接复制");
            mat.copyTo(outputMat);
        } else {
            console.log("不支持的通道数: " + channels);
            outputMat.release();
            return null;
        }

        // 检查转换后的Mat
        if (isMatEmpty(outputMat)) {
            console.log("颜色转换后Mat为空或无效");
            outputMat.release();
            return null;
        }

        // 创建Bitmap
        log("创建Bitmap");
        bmp = Bitmap.createBitmap(outputMat.cols(), outputMat.rows(), Bitmap.Config.ARGB_8888);

        if (!bmp) {
            console.log("Bitmap创建失败");
            outputMat.release();
            return null;
        }

        // Mat转Bitmap
        Utils.matToBitmap(outputMat, bmp);
        log("Mat转Bitmap完成");

        return bmp;

    } catch (e) {
        console.log("Mat转Bitmap异常: " + e.message);
        return null;
    } finally {
        // 确保释放临时Mat
        if (outputMat && !isMatEmpty(outputMat)) {
            outputMat.release();
        }
    }
}

/**
 * Image输出处理
 */
function handleImageOutput(input) {
    var bitmap = null;
    var needsRecycle = false;
    var baos = null;

    try {
        if (input instanceof Bitmap) {
            if (input.isRecycled()) {
                console.log("输入Bitmap已被回收");
                return images.createImage(1, 1);
            }
            bitmap = input;
        } else if (input instanceof Mat) {
            bitmap = matToBitmap(input);
            needsRecycle = true;
            if (!bitmap) {
                return images.createImage(1, 1);
            }
        } else if (input && typeof input.getBitmap === 'function') {
            return input
        }

        if (!bitmap) {
            console.log("无法转换到Image: 不支持的输入类型");
            return images.createImage(1, 1);
        }

        // 转换为字节数组
        baos = new java.io.ByteArrayOutputStream();
        var success = bitmap.compress(Bitmap.CompressFormat.PNG, 100, baos);

        if (!success) {
            console.log("Bitmap压缩失败");
            return images.createImage(1, 1);
        }

        var bytes = baos.toByteArray();
        baos.close();
        baos = null;

        if (!bytes || bytes.length === 0) {
            console.log("字节数组为空");
            return images.createImage(1, 1);
        }

        var resultImage = images.fromBytes(bytes);

        return resultImage || images.createImage(1, 1);

    } catch (e) {
        console.log("Image转换异常: " + e.message);
        return images.createImage(1, 1);
    } finally {
        // 清理资源
        if (needsRecycle && bitmap && !bitmap.isRecycled()) {
            bitmap.recycle();
        }
        try {
            if (baos) baos.close();
        } catch (e) {
            // 忽略关闭异常
        }
    }
}

/**
 * 改进的输出处理函数
 */
function handleOutput(input, type) {
    type = type || "img";

    // 输入验证
    if (!input) {
        console.log("输入为空或未定义");
        return getDefaultOutput(type);
    }

    try {
        switch (type) {
            case "mat":
                return handleMatOutput(input);
            case "bitmap":
                return handleBitmapOutput(input);
            default:
                return handleImageOutput(input);
        }
    } catch (e) {
        console.log("输出处理异常: " + e.message);
        console.log("堆栈: " + e.stack);
        return getDefaultOutput(type);
    }
}

/**
 * 应用单个滤镜（简化版）
 */
function applySingleFilter(src, dst, mode, config) {
    if (!src || isMatEmpty(src) || !dst) return false;

    var temp = new Mat();

    try {
        switch (mode) {
            case "grayscale":
                if (src.channels() > 1) {
                    safeColorConvert(src, dst, Imgproc.COLOR_BGR2GRAY);
                } else {
                    src.copyTo(dst);
                }
                break;

            case "binary":
                if (src.channels() > 1) {
                    safeColorConvert(src, temp, Imgproc.COLOR_BGR2GRAY);
                } else {
                    src.copyTo(temp);
                }
                Imgproc.threshold(temp, dst, config.threshold, config.maxValue, config.type);
                break;

            case "adaptive_binary":
                if (src.channels() > 1) {
                    safeColorConvert(src, temp, Imgproc.COLOR_BGR2GRAY);
                } else {
                    src.copyTo(temp);
                }
                Imgproc.adaptiveThreshold(temp, dst, config.maxValue,
                    config.adaptiveMethod, config.thresholdType,
                    config.blockSize, config.C);
                break;

            case "edge":
                if (src.channels() > 1) {
                    safeColorConvert(src, temp, Imgproc.COLOR_BGR2GRAY);
                } else {
                    src.copyTo(temp);
                }
                Imgproc.Canny(temp, dst, config.threshold1, config.threshold2, config.apertureSize, config.L2gradient);
                break;

            case "gaussian_blur":
                var kSize = Math.max(1, config.kernelSize);
                if (kSize % 2 === 0) kSize++;
                Imgproc.GaussianBlur(src, dst, new Size(kSize, kSize), config.sigmaX, config.sigmaY);
                break;

            case "box_blur":
                var kSize = Math.max(1, config.kernelSize);
                Imgproc.blur(src, dst, new Size(kSize, kSize));
                break;

            case "median_blur":
                var kSize = Math.max(1, config.kernelSize);
                if (kSize % 2 === 0) kSize++;
                Imgproc.medianBlur(src, dst, kSize);
                break;

            case "high_contrast":
                src.convertTo(dst, -1, config.alpha, config.beta);
                break;

            case "brightness":
                src.convertTo(dst, -1, 1.0, config.value);
                break;

            case "sketch":
                applySketchFilter(src, dst, config);
                break;

            case "erosion":
                applyMorphologyOperation(src, dst, Imgproc.MORPH_ERODE, config);
                break;

            case "dilation":
                applyMorphologyOperation(src, dst, Imgproc.MORPH_DILATE, config);
                break;

            case "opening":
                applyMorphologyOperation(src, dst, Imgproc.MORPH_OPEN, config);
                break;

            case "closing":
                applyMorphologyOperation(src, dst, Imgproc.MORPH_CLOSE, config);
                break;
            case "replacecolor":
                applyColorReplace(src, dst, config);
                break;

            default:
                src.copyTo(dst);
                break;
        }
        return true;
    } catch (e) {
        console.log("滤镜应用失败 [" + mode + "]: " + e);
        try {
            src.copyTo(dst);
        } catch (e2) {
            console.log("复制原图失败: " + e2);
        }
        return false;
    } finally {
        if (temp && !isMatEmpty(temp)) temp.release();
    }
}

/**
 * 素描滤镜（简化版）
 */
function applySketchFilter(src, dst, config) {
    var gray = new Mat();
    var edges = new Mat();

    try {
        if (src.channels() > 1) {
            safeColorConvert(src, gray, Imgproc.COLOR_BGR2GRAY);
        } else {
            src.copyTo(gray);
        }

        // 使用自适应阈值创建素描效果
        Imgproc.adaptiveThreshold(gray, edges, 255,
            Imgproc.ADAPTIVE_THRESH_MEAN_C,
            Imgproc.THRESH_BINARY, 9, 9);

        // 反转颜色
        Core.bitwise_not(edges, edges);

        if (src.channels() > 1) {
            safeColorConvert(edges, dst, Imgproc.COLOR_GRAY2BGR);
        } else {
            edges.copyTo(dst);
        }

    } catch (e) {
        console.log("素描滤镜失败: " + e);
        src.copyTo(dst);
    } finally {
        if (!isMatEmpty(gray)) gray.release();
        if (!isMatEmpty(edges)) edges.release();
    }
}

/**
 * 形态学操作
 */
function applyMorphologyOperation(src, dst, operation, config) {
    var kernel = getMorphologyKernel(config.kernelSize, config.shape);

    try {
        Imgproc.morphologyEx(src, dst, operation, kernel,
            new Point(-1, -1), config.iterations);
    } finally {
        if (kernel) kernel.release();
    }
}

/**
 * 颜色替换
 */
function applyColorReplace(src, dst, config) {
    // 确保输入是有效的 BGR 图像
    var workingSrc = new Mat();
    if (src.channels() === 4) {
        Imgproc.cvtColor(src, workingSrc, Imgproc.COLOR_RGBA2BGR);
    } else {
        src.copyTo(workingSrc);
    }

    var processedMask = new Mat(workingSrc.rows(), workingSrc.cols(), CvType.CV_8UC1, new Scalar(0));
    var result = workingSrc.clone();
    var tempMask = new Mat();

    function hexToScalar(hex) {
        if (!hex || hex === "" || typeof hex !== 'string') return null;
        var cleanHex = hex.replace("#", "");
        if (cleanHex.length !== 6) return null;
        var r = parseInt(cleanHex.substring(0, 2), 16);
        var g = parseInt(cleanHex.substring(2, 4), 16);
        var b = parseInt(cleanHex.substring(4, 6), 16);
        // 显式创建双精度数组确保 Java 识别
        return new Scalar([b, g, r]); 
    }

    if (config.colors && Array.isArray(config.colors)) {
        config.colors.forEach(function(item, idx) {
            var findS = hexToScalar(item.find);
            if (!findS) return;

            var changeS = hexToScalar(item.change);
            var sim = parseFloat(item.similarity || 1.0);
            // 修正：如果相似度为 0.95，容差应覆盖约 13 个单位 (255 * 0.05)
            var tolerance = Math.max(0, Math.floor(255 * (1.001 - sim)));

            var b = findS.val[0], g = findS.val[1], r = findS.val[2];
            
            var lower = new Scalar(Math.max(b - tolerance, 0), Math.max(g - tolerance, 0), Math.max(r - tolerance, 0));
            var upper = new Scalar(Math.min(b + tolerance, 255), Math.min(g + tolerance, 255), Math.min(r + tolerance, 255));

            Core.inRange(workingSrc, lower, upper, tempMask);
            
            // 调试用：查看当前颜色匹配到了多少像素
            var count = Core.countNonZero(tempMask);

            if (count > 0 && changeS) {
                result.setTo(changeS, tempMask);
            }
            
            Core.bitwise_or(processedMask, tempMask, processedMask);
        });
    }

    // 处理 other 逻辑
    var otherS = hexToScalar(config.other);
    if (otherS) {
        var unmatchedMask = new Mat();
        Core.bitwise_not(processedMask, unmatchedMask);
        result.setTo(otherS, unmatchedMask);
        unmatchedMask.release();
    }

    // 如果原图是 RGBA，转回去
    if (src.channels() === 4) {
        Imgproc.cvtColor(result, dst, Imgproc.COLOR_BGR2RGBA);
    } else {
        result.copyTo(dst);
    }

    // 释放资源
    processedMask.release();
    result.release();
    tempMask.release();
    workingSrc.release();
}

/**
 * 滤镜流水线（简化版）
 */
function 滤镜流水线(img, filters, outputType) {
    if (!img) {
        console.log("输入图像为空");
        return null;
    }

    // 空滤镜直接返回
    if (!filters || (Array.isArray(filters) && filters.length === 0)) {
        return handleOutput(img, outputType);
    }
    if (!Array.isArray(filters)) filters = [filters];

    var src = new Mat();
    var dst = new Mat();
    var temp = new Mat();

    try {
        // 处理不同类型的输入
        if (img instanceof Mat) {
            src = img.clone();
        } else if (img instanceof Bitmap) {
            Utils.bitmapToMat(img, src);
        } else if (typeof img.getBitmap === 'function') {
            Utils.bitmapToMat(img.getBitmap(), src);
        } else {
            console.log("不支持的输入类型");
            return handleOutput(img, outputType);
        }

        if (isMatEmpty(src)) {
            console.log("Mat转换失败");
            return handleOutput(img, outputType);
        }

        console.log("图像信息: " + src.cols() + "x" + src.rows() + ", 通道数: " + src.channels());
        src.copyTo(dst);

        // 依次应用每个滤镜
        for (var i = 0; i < filters.length; i++) {
            var filter = filters[i];
            if (!filter || !filter.name) continue;

            var mode = filterMap[filter.name] || filter.name;
            if (!defaultConfigs[mode]) {
                console.log("未知滤镜: " + filter.name);
                continue;
            }

            var config = {};
            Object.assign(config, defaultConfigs[mode]);
            if (filter.config) Object.assign(config, filter.config);

            console.log("应用滤镜: " + filter.name + " (" + mode + ")");

            dst.copyTo(temp);
            var success = applySingleFilter(temp, dst, mode, config);
            if (!success) {
                console.log("滤镜应用失败: " + filter.name + "，使用上一步结果");
                temp.copyTo(dst);
            }
        }

        return handleOutput(dst, outputType);

    } catch (e) {
        console.log("滤镜流水线异常: " + e);
        return handleOutput(img, outputType);
    } finally {
        try {
            if (src && !isMatEmpty(src)) src.release();
            if (dst && !isMatEmpty(dst)) dst.release();
            if (temp && !isMatEmpty(temp)) temp.release();
        } catch (e) {
            console.log("资源释放异常: " + e);
        }
    }
}

/**
 * 获取所有可用滤镜列表
 */
function 获取滤镜列表() {
    return Object.keys(filterMap);
}

/**
 * 获取滤镜分类（移除反色和锐化）
 */
function 获取滤镜分类() {
    return {
        "基础滤镜": ["灰度滤镜", "黑白滤镜", "自适应二值化", "轮廓滤镜"],
        "模糊效果": ["高斯模糊", "均值模糊", "中值模糊"],
        "色彩调整": ["高对比度", "亮度调整"],
        "特殊效果": ["素描效果"],
        "形态学操作": ["腐蚀", "膨胀", "开运算", "闭运算"],
    };
}

/**
 * 获取滤镜的默认配置
 */
function 获取滤镜配置(filterName) {
    var mode = filterMap[filterName];
    return mode ? JSON.parse(JSON.stringify(defaultConfigs[mode])) : {};
}

/**
 * 简化版预设组合（移除包含反色和锐化的预设）
 */
var 预设组合 = {
    // 找图优化系列
    "找图增强_通用": [{
            name: "中值模糊",
            config: {
                kernelSize: 3
            }
        },
        {
            name: "高对比度",
            config: {
                alpha: 1.3,
                beta: 0
            }
        }
    ],

    "找图增强_暗图": [{
            name: "亮度调整",
            config: {
                value: 40
            }
        },
        {
            name: "高对比度",
            config: {
                alpha: 1.4,
                beta: 10
            }
        }
    ],

    "找图增强_亮图": [{
            name: "高对比度",
            config: {
                alpha: 1.2,
                beta: -10
            }
        },
        {
            name: "中值模糊",
            config: {
                kernelSize: 3
            }
        }
    ],

    "找图增强_低对比": [{
        name: "高对比度",
        config: {
            alpha: 1.6,
            beta: 0
        }
    }],

    "找图增强_文字识别": [{
            name: "灰度滤镜"
        },
        {
            name: "高对比度",
            config: {
                alpha: 1.4,
                beta: 10
            }
        },
        {
            name: "自适应二值化",
            config: {
                blockSize: 11,
                C: 2
            }
        },
        {
            name: "膨胀",
            config: {
                kernelSize: 3,
                iterations: 1
            }
        }
    ],

    // 美化系列
    "简单美化": [{
        name: "高对比度",
        config: {
            alpha: 1.2,
            beta: 10
        }
    }],

    "黑白艺术": [{
            name: "灰度滤镜"
        },
        {
            name: "高对比度",
            config: {
                alpha: 1.4,
                beta: 0
            }
        }
    ],

    "清晰增强": [{
            name: "中值模糊",
            config: {
                kernelSize: 3
            }
        },
        {
            name: "高对比度",
            config: {
                alpha: 1.1,
                beta: 5
            }
        }
    ],

    "素描风格": [{
            name: "素描效果"
        },
        {
            name: "高对比度",
            config: {
                alpha: 1.3,
                beta: 20
            }
        }
    ]
};

/**
 * 应用预设组合
 */
function 应用预设(img, presetName) {
    if (!预设组合[presetName]) {
        console.log("未找到预设: " + presetName);
        return img.clone();
    }

    console.log("应用预设组合: " + presetName);
    return 滤镜流水线(img, 预设组合[presetName]);
}

/**
 * 获取预设列表
 */
function 获取预设列表() {
    return Object.keys(预设组合);
}

/**
 * 测试滤镜
 */
function 测试滤镜(img, filterName, config) {
    try {
        var filter = {
            name: filterName
        };
        if (config) filter.config = config;

        var result = 滤镜流水线(img, filter);
        return result !== null;
    } catch (e) {
        console.log("滤镜测试异常 [" + filterName + "]: " + e);
        return false;
    }
}

module.exports = {
    滤镜流水线,
    获取滤镜列表,
    获取滤镜分类,
    获取滤镜配置,
    测试滤镜,
    应用预设,
    获取预设列表,
    filterMap,
    defaultConfigs,
    预设组合
};