"nodejs";

/**
 * 优化重点：
 * 1. 严格执行“谁最后用图谁回收”原则，杜绝读取已回收图像。
 * 2. 优化 Promise.race 的清理机制，确保超时后后台截图任务立即停止。
 * 3. 修正 cycleRun 的异步执行流，确保逻辑顺序正确。
 */

const delay = (ms) => new Promise(resolve => setTimeout(resolve, ms));

/**
 * 闪电响应版：等待分数变化
 */
async function whileScorePointsChange(baseScore, maxWaitTime) {
    let isFinished = false;
    const startTime = Date.now();

    // 任务 A：绝对超时控制
    const timeoutPromise = (async () => {
        await delay(maxWaitTime);
        if (!isFinished) {
            isFinished = true; // 强制通知识别循环停止
            return false;
        }
    })();

    // 任务 B：后台高频识别
    const monitorPromise = (async () => {
        try {
            while (!isFinished && (Date.now() - startTime < maxWaitTime)) {
                const img = await capturer.nextImage();
                if (!img || img.isRecycled) {
                    await delay(5);
                    continue;
                }

                // 重点：如果已经超时，拿到图立刻扔掉，不再计算
                if (isFinished) {
                    img.recycle();
                    break;
                }

                const currentSP = await getScorePoints(img);
                img.recycle(); // 识别完立即回收

                if (currentSP !== baseScore) {
                    isFinished = true;
                    return true;
                }
                await delay(5); // 给主线程留出 5ms 响应点击
            }
        } catch (e) {
            console.error("监控循环异常:", e);
        }
        return false;
    })();

    const result = await Promise.race([timeoutPromise, monitorPromise]);
    isFinished = true; // 确保所有任务都感知到结束
    return !!result;
}

/**
 * 极速流水线循环
 */
async function cycleRun() {
    if (isRunning) return;
    isRunning = true;
    console.log("🚀 Node.js 极速流水线已启动");

    while (isRunning) {
        let img = null;
        try {
            // 1. 获取最新帧
            img = await capturer.nextImage();
            if (!img || img.isRecycled) {
                await delay(50);
                continue;
            }

            // 2. 核心数据识别 (并行计算)
            const data = await getAllData(img);
            if (!data || !data.thorns) {
                img.recycle();
                await delay(50);
                continue;
            }

            // 3. 处理复活逻辑
            if (data.thorns.length === 0) {
                loopCount++;
                if (loopCount >= RESURGENCE_CHECK_INTERVAL) {
                    loopCount = 0;
                    serverEngine.emit('getResurgenceButton');
                }
                img.recycle();
                await delay(200);
                continue;
            }
            loopCount = 0;

            // 4. 计算跳跃参数
            const endX = data.endX;
            const pressTime = await ckltJumpToXTime(endX) / (global.runSpeed || 1);
            
            // 5. 执行非阻塞指令 (绘制 & 跳跃)
            if (endX > 0) {
                // 异步执行，不 await 它们，让主循环先行
                jumpToX(endX, pressTime); 
                serverEngine.emit('drawScreenImg', {
                    data: data,
                    imgSizes: { width: img.width, height: img.height }
                });
            }

            // ⚠️ 关键优化：识别完这张图后，直接回收！
            // 这样 whileScorePointsChange 内部可以开辟全新的截图流，互不干扰
            const baseScore = data.scorePoints; 
            img.recycle();
            img = null; // 彻底切断引用

            // 6. 等待控制
            if (global.whileScoreChangeBoor) {
                // 进入全速监控模式 (由于上面已 recycle，这里不会报错)
                await whileScorePointsChange(baseScore, global.waitTime / (global.runSpeed || 1));
            } else {
                // 传统等待模式
                let sleepTime = pressTime * 2;
                sleepTime = Math.max(10, Math.min(sleepTime, 5000));
                await delay(sleepTime);
                if (sleepTime <= 450) await delay(68);
            }

        } catch (e) {
            console.error("cycleRun 核心循环崩溃:", e);
            if (img && !img.isRecycled) img.recycle();
            await delay(500); 
        }
    }
}

/**
 * 闪电响应版：等待分数变化
 * 使用 Promise.race 确保超时立即中断，不浪费 1 毫秒
 */
async function whileScorePointsChange(scorePoints, maxWaitTime) {
    console.log(`[实时监控] 启动! 基准分数: ${scorePoints}, 绝对限时: ${maxWaitTime}ms`);

    let isFinished = false; // 控制开关

    // 1. 超时计时器：时间一到准时发令
    const timeoutPromise = (async () => {
        await delay(maxWaitTime);
        if (!isFinished) {
            console.log("[超时] 强制中断，立即返回");
            return false;
        }
    })();

    // 2. 识别循环：在后台拼命识别
    const monitorPromise = (async () => {
        const startTime = Date.now();
        try {
            while (Date.now() - startTime < maxWaitTime && !isFinished) {
                const img = await capturer.nextImage();
                if (!img || img.isRecycled) {
                    await delay(2);
                    continue;
                }

                // 如果此时已经超时了，直接回收并退出，不再识别
                if (isFinished) {
                    img.recycle();
                    break;
                }

                const currentSP = await getScorePoints(img);
                img.recycle();

                if (currentSP !== scorePoints) {
                    isFinished = true; // 标记结束，让超时计时器失效
                    return true;
                }

                // 极致响应：在 Node.js 中 delay(0) 可以让出执行权但不产生多余等待
                await delay(2);
            }
        } catch (e) {
            console.error("识别循环异常:", e);
        }
        return false;
    })();

    // 3. 赛跑：谁快听谁的
    const result = await Promise.race([timeoutPromise, monitorPromise]);
    isFinished = true; // 确保所有逻辑停止
    return !!result;
}



/**
 * 脚本主循环 - 异步流水线版
 * * 流程复盘：
 * 1. 采集：使用 nextImage 瞬间获取内存图像。
 * 2. 识别：通过 getAllData 并行计算荆棘和分数。
 * 3. 决策：不阻塞主流程，直接发出 jumpToX 指令。
 * 4. 视觉：异步绘制并自动管理内存回收。
 */
async function mainRun() {
    let capture = null;

    try {
        // 1. 获取最新帧
        capture = await capturer.nextImage();
        if (!capture || capture.isRecycled) return false;

        // 2. 核心识别
        const startTime = performance.now();
        const data = await getAllData(capture);
        const calcTime = (performance.now() - startTime).toFixed(2);

        // 3. 异步并发层 (动作执行与 UI 反馈)
        // 注意：末尾加了 () 确保它立即执行
        (async () => {
            try {
                const {
                    endX,
                    pressTime,
                    thorns
                } = data;
                const actualPressTime = pressTime / (global.runSpeed || 1);

                // --- A. 物理动作层 ---
                if (endX && endX > 0) {
                    jumpToX(endX, actualPressTime);
                }

                // --- B. 视觉绘制层 ---
                if (thorns && thorns.length > 0) {
                    // 发送给 UI 线程进行绘制
                    serverEngine.emit('drawScreenImg', {
                        data: data,
                        imgSizes: {
                            width: capture.width,
                            height: capture.height
                        }

                    });
                }
            } catch (innerError) {
                console.error("异步执行层异常:", innerError);
            } finally {
                // --- C. 资源释放层 ---
                // 在异步逻辑执行完（或报错后）回收图片
                if (capture && !capture.isRecycled) {
                    capture.recycle();
                }
            }
        })(); // <--- 关键：加了括号才会立即执行

        return true;

    } catch (e) {
        console.error("mainRun 运行异常:", e);
        // 如果第一步就挂了，也要回收
        if (capture && !capture.isRecycled) capture.recycle();
        return false;
    }
}


// 创建一个控制信号，模拟原版的 cycleRun.state
let isRunning = false;
let loopCount = 0;
const RESURGENCE_CHECK_INTERVAL = 3;

/**
 * 辅助延迟函数 (替代 sleep)
 */
// const delay = (ms) => new Promise(resolve => setTimeout(resolve, ms));
// 循环执行  --2026-2-11 22:46 新增
async function cycleRun() {
    if (isRunning) return;
    isRunning = true;
    console.log("🚀 Node.js 引擎主循环已启动");

    while (isRunning) {
        let img = null;
        try {
            // 1. 获取截图 (假设 capturer 已在外部初始化)
            // 如果你没有 capturer，可以用 images.captureScreen()
            img = await capturer.nextImage();
            
            if (!img || img.isRecycled) {
                console.error("截图失败，重试中...");
                await delay(100);
                continue;
            }

            // 2. 核心数据识别
            const data = await getAllData(img);
            if (!data || !data.thorns) {
                console.warn("数据识别异常");
                img.recycle();
                await delay(50);
                continue;
            }

            // --- 复活逻辑处理 ---
            if (data.thorns.length === 0) {
                loopCount++;
                if (loopCount >= RESURGENCE_CHECK_INTERVAL) {
                    loopCount = 0;
                    serverEngine.emit('getResurgenceButton');
                    await delay(0);
                }
                img.recycle();
                await delay(100);
                // continue;
            }
            loopCount = 0; // 识别到荆棘，重置计数器

            // 3. 计算跳跃参数
            const endX = data.endX;
            // 假设 ckltJumpToXTime 是你定义的计算函数
            const pressTime = await ckltJumpToXTime(endX) / (runSpeed || 1);
            let sleepTime = pressTime * 2;

            // 4. 执行跳跃 (非阻塞)
            if (endX && endX > 0) {
                // 直接调用，不 await，让它异步去按压
                await jumpToX(endX, pressTime);
            }

            // 5. 效果绘制 (非阻塞并发)
            // 在 Node.js 中，直接发起一个异步函数即可，不需要 threads.start
            // --- B. 视觉绘制层 ---
            if (data.thorns && data.thorns.length > 0) {
                // 发送给 UI 线程进行绘制
                serverEngine.emit('drawScreenImg', {
                    data: data,
                    imgSizes: {
                        width: img.width,
                        height: img.height
                    }

                });
            }

            // 6. 智能等待控制
            if (endX >= 0) {
                sleepTime = Math.max(10, Math.min(sleepTime, 5000));
                await delay(sleepTime);
                if (sleepTime <= 450) await delay(68);
            } else {
                await delay(10);
            }

            img.recycle();

            // 分数变化检测 (异步等待)
            if (whileScoreChangeBoor) {
                const scorePoints = getScorePoints(img);
                await whileScorePointsChange(data.scorePoints, waitTime / (runSpeed || 1));


            } else {
                await delay(waitTime);

            }
            // 等待一帧
            // await delay(0);

        } catch (e) {
            console.error("cycleRun 核心循环崩溃:", e);
            if (img && !img.isRecycled) img.recycle();
            await delay(500); // 崩溃后强制冷静半秒，防止死循环死机
        }
        // 注意：在 Node.js 中，img 的回收在异步绘制层处理，或者在这里统一处理
    }
}

// 停止函数
function stopCycle() {
    isRunning = false;
    console.log("🛑 循环停止信号已发出");
}