// 线程池管理器
var ThreadPoolManager = (function() {
    var instance;

    function init() {
        var pools = {};

        return {
            // 获取或创建线程池
            getPool: function(name, config) {
                if (!pools[name]) {
                    pools[name] = $threads.pool({
                        corePoolSize: config.corePoolSize || 4,
                        maxPoolSize: config.maxPoolSize || 8
                    });
                }
                return pools[name];
            },

            // 关闭所有线程池
            shutdownAll: function() {
                for (var poolName in pools) {
                    try {
                        if (!pools[poolName].isShutdown()) {
                            pools[poolName].shutdown();
                        }
                    } catch (e) {
                        console.error("关闭线程池[" + poolName + "]异常:", e);
                    }
                }
                pools = {};
            }
        };
    }

    return {
        getInstance: function() {
            if (!instance) {
                instance = init();
            }
            return instance;
        }
    };
})();

// 任务处理器基类
function BaseTaskHandler() {
    this.beforeAll = function() {}; // 所有任务执行前的预处理
    this.afterAll = function() {}; // 所有任务完成后的处理
    this.processTask = function(task, index) { // 单个任务处理逻辑
        throw new Error("必须实现processTask方法");
    };
    this.shouldContinue = function(results) { // 是否继续处理剩余任务
        return true;
    };
}

// 多线程任务执行器
function MultiThreadExecutor(poolName, poolConfig) {
    this.pool = ThreadPoolManager.getInstance().getPool(poolName, poolConfig || {});
    this.latch = null;
    this.lock = new java.util.concurrent.locks.ReentrantLock();
    this.results = [];
    this.stopFlag = new java.util.concurrent.atomic.AtomicBoolean(false);

    // 执行任务
    this.execute = function(tasks, handler) {
        if (!handler || typeof handler.processTask !== "function") {
            throw new Error("必须提供有效的任务处理器");
        }

        // 初始化
        this.results = [];
        this.stopFlag.set(false);
        this.latch = new java.util.concurrent.CountDownLatch(tasks.length);

        // 预处理
        handler.beforeAll();

        var self = this;

        // 提交所有任务到线程池
        tasks.forEach(function(task, index) {
            self.pool.execute(function() {
                try {
                    // 检查是否应该停止处理
                    if (self.stopFlag.get()) {
                        return;
                    }

                    // 处理单个任务
                    var result = handler.processTask(task, index);
                    if (result) {
                        self.lock.lock();
                        try {
                            // 检查是否应该停止处理
                            if (self.stopFlag.get()) {
                                return;
                            }
                            self.results.push(result);

                            // 检查是否应该停止处理剩余任务
                            if (!handler.shouldContinue(self.results)) {
                                self.stopFlag.set(true);
                            }
                        } finally {
                            self.lock.unlock();
                        }
                    }
                } catch (e) {
                    console.error("任务处理异常:", e);
                } finally {
                    self.latch.countDown();
                }
            });
        });

        // 等待所有任务完成
        this.latch.await();

        // 后处理
        handler.afterAll();

        // 返回结果
        return this.results;
    };
}

// 脚本退出清理资源
events.on("exit", function() {
    ThreadPoolManager.getInstance().shutdownAll();
});


module.exports = {
    ThreadPoolManager,
    BaseTaskHandler,
    MultiThreadExecutor
}