function ImageCache(dir, maxMemory) {
    this._imgCache = {};
    this._accessTime = {};
    this.dir = dir || files.join(files.cwd(), "images");
    this._maxMemory = maxMemory || 200 * 1024 * 1024; // 最大允许占用内存（单位：字节）
}
//获取图片
ImageCache.prototype.get = function(k, extension) {
    extension = extension || "png";

    var imgObj = this._imgCache[k]
    this._accessTime[k] = Date.now(); // 每次访问或设置后记录时间
    if (imgObj && !imgObj.isRecycled()) {
        return imgObj
    }
    var path = files.join(this.dir, k + "." + extension)
    if (!files.exists(path)) {
        throw "图片不存在"
    }
    imgObj = images.read(path)
    if (!imgObj) {
        throw "读取图片失败"
    }
    this._imgCache[k] = imgObj
    this._ensureMemory(); // 在缓存插入后调用
    return imgObj
}
//设置图片
ImageCache.prototype.set = function(k, v) {
    var imgCache = this._imgCache
    if (/^Image/.test(v.toString()) && v != imgCache[k]) {
        this._accessTime[k] = Date.now(); // 每次访问或设置后记录时间
        imgCache[k] && !imgCache[k].isRecycled() && imgCache[k].recycle()
        imgCache[k] = v
        this._ensureMemory(); // 在缓存插入后调用
        return true
    }
    return false
}
//删除图片
ImageCache.prototype.delete = function(k) {
    if (this._imgCache[k]) {
        this._imgCache[k].recycle()
        delete this._imgCache[k]
        delete this._accessTime[k]
        return true
    } else {
        return false
    }
}
//清空图片
ImageCache.prototype.clear = function() {
    var imgCache = this._imgCache
    for (k in imgCache) {
        if (!imgCache[k].isRecycled()) {
            imgCache[k].recycle()
        }
    }
    this._imgCache = {}
    this._accessTime = {}
    return true
}
//重新读取图片
ImageCache.prototype.reread = function(k, extension) {
    extension = extension || "png";
    var path = files.join(this.dir, k + "." + extension)
    if (!files.exists(path)) {
        throw "图片不存在"
    }
    if (this._imgCache[k]) {
        this._imgCache[k].recycle()
        delete this._imgCache[k]
    }
    imgObj = images.read(path)
    if (!imgObj) {
        throw "读取图片失败"
    }
    this._imgCache[k] = imgObj
    this._accessTime[k] = Date.now(); // 每次访问或设置后记录时间
    this._ensureMemory(); // 在缓存插入后调用
    return imgObj
}
//获取所有图片是否被回收
ImageCache.prototype.stats = function() {
    var arr = []
    var imgCache = this._imgCache
    for (var k in imgCache) {
        arr.push({
            name: k,
            recycled: imgCache[k].isRecycled()
        })
    }
    return arr
}
// 图片存在吗
ImageCache.prototype.has = function(k) {
    if (k) {
        var img = this._imgCache[k]
        if (img) {
            return {
                name: k,
                isRecycled: img.isRecycled()
            }
        }
        return null
    }
    throw "请输入图片名"
}

// 图片缓存数量
ImageCache.prototype.size = function() {
    return Object.keys(this._imgCache).length;
}

// 获取图片内存占用
ImageCache.prototype.memoryEstimate = function(type) {
    var total = 0;
    var imgCache = this._imgCache;
    for (var k in imgCache) {
        var img = imgCache[k];
        if (!img.isRecycled()) {
            total += img.getWidth() * img.getHeight() * 4; // 每像素4字节（RGBA）
        }
    }
    switch (type) {
        case "MB":
            return (total / 1024 / 1024).toFixed(2) + ' MB';
        default:
            return total
    }
}

//按内存回收
ImageCache.prototype._ensureMemory = function() {
    if (!this._maxMemory) return;

    var currentMemory = this.memoryEstimate();
    if (currentMemory <= this._maxMemory) return;

    var keys = Object.keys(this._imgCache);
    keys.sort((a, b) => (this._accessTime[a] || 0) - (this._accessTime[b] || 0)); // 按最久未访问排序

    for (var i = 0; i < keys.length && currentMemory > this._maxMemory; i++) {
        var k = keys[i];
        var img = this._imgCache[k];
        if (!img.isRecycled()) {
            currentMemory -= img.getWidth() * img.getHeight() * 4;
            img.recycle();
            delete this._imgCache[k];
            delete this._accessTime[k];
        }
    }
}

// 手动更新使用时间避免被回收
ImageCache.prototype.touch = function(k) {
    if (this._imgCache[k]) {
        this._accessTime[k] = Date.now();
        return true;
    }
    return false;
}
//设置图片内存占用阈值
ImageCache.prototype.setMaxMemory = function(bytes) {
    this._maxMemory = bytes;
}

module.exports = ImageCache